<?php
/**
 * The Global functionality of the plugin.
 *
 * Defines the functionality loaded on admin.
 *
 * @since      1.0.49
 * @package    RankMath
 * @subpackage RankMath\Rest
 * @author     Rank Math <support@rankmath.com>
 */

namespace RankMath\Instant_Indexing;

use WP_Error;
use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Controller;
use WP_REST_Response;
use MyThemeShop\Helpers\Arr;

defined( 'ABSPATH' ) || exit;

/**
 * Rest class.
 */
class Rest extends WP_REST_Controller {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->namespace = \RankMath\Rest\Rest_Helper::BASE . '/in';
	}

	/**
	 * Register REST routes.
	 *
	 * @return void
	 */
	public function register_routes() {
		$namespace = $this->namespace;

		$endpoint = '/submitUrls/';
		register_rest_route(
			$namespace,
			$endpoint,
			[
				[
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => [ $this, 'submit_urls' ],
					'permission_callback' => [ '\\RankMath\\Rest\\Rest_Helper', 'can_manage_options' ],
				],
			]
		);

		$endpoint = '/getLog/';
		register_rest_route(
			$namespace,
			$endpoint,
			[
				[
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => [ $this, 'get_log' ],
					'permission_callback' => [ '\\RankMath\\Rest\\Rest_Helper', 'can_manage_options' ],
				],
			]
		);

		$endpoint = '/clearLog/';
		register_rest_route(
			$namespace,
			$endpoint,
			[
				[
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => [ $this, 'clear_log' ],
					'permission_callback' => [ '\\RankMath\\Rest\\Rest_Helper', 'can_manage_options' ],
				],
			]
		);
	}

	/**
	 * Submit URLs.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function submit_urls( WP_REST_Request $request ) {
		$urls = $request->get_param( 'urls' );
		if ( empty( $urls ) ) {
			return new WP_Error( 'empty_urls', __( 'No URLs provided.', 'rank-math' ) );
		}

		$urls = Arr::from_string( $urls, "\n" );
		$urls = array_values( array_unique( array_filter( $urls, 'wp_http_validate_url' ) ) );

		if ( ! $urls ) {
			return new WP_Error( 'invalid_urls', __( 'Invalid URLs provided.', 'rank-math' ) );
		}

		$result = Api::get()->submit( $urls );
		if ( ! $result ) {
			return new WP_Error( 'submit_failed', __( 'Failed to submit URLs. See details in the History tab.', 'rank-math' ) );
		}

		$urls_number = count( $urls );
		return new WP_REST_Response(
			[
				'success' => true,
				'message' => sprintf(
					// Translators: %s is the number of URLs submitted.
					_n(
						'Successfully submitted %s URL.',
						'Successfully submitted %s URLs.',
						$urls_number,
						'rank-math'
					),
					$urls_number
				),
			]
		);
	}

	/**
	 * Get log.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function get_log( WP_REST_Request $request ) {
		$filter = $request->get_param( 'filter' );
		$result = Api::get()->get_log();
		$total = count( $result );
		foreach ( $result as $key => $value ) {
			$result[ $key ]['timeFormatted'] = wp_date( 'Y-m-d H:i:s', $value['time'] );
			// Translators: placeholder is human-readable time, e.g. "1 hour".
			$result[ $key ]['timeHumanReadable'] = sprintf( __( '%s ago', 'rank-math' ), human_time_diff( $value['time'] ) );

			if ( 'manual' === $filter && empty( $result[ $key ]['manual_submission'] ) ) {
				unset( $result[ $key ] );
			} elseif ( 'auto' === $filter && ! empty( $result[ $key ]['manual_submission'] ) ) {
				unset( $result[ $key ] );
			}
		}

		$result = array_values( array_reverse( $result ) );

		return new WP_REST_Response(
			[
				'data'  => $result,
				'total' => $total,
			]
		);
	}

	/**
	 * Clear log.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function clear_log( WP_REST_Request $request ) {
		Api::get()->clear_log();
		return new WP_REST_Response( [ 'status' => 'ok' ] );
	}
}
